import { Navbar } from "@/components/layout/Navbar";
import { Footer } from "@/components/layout/Footer";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Textarea } from "@/components/ui/textarea";
import { Card, CardContent, CardHeader } from "@/components/ui/card";
import { Accordion, AccordionContent, AccordionItem, AccordionTrigger } from "@/components/ui/accordion";
import { Mail, Phone, MapPin, Send } from "lucide-react";
import { useForm } from "react-hook-form";
import { z } from "zod";
import { zodResolver } from "@hookform/resolvers/zod";
import { Form, FormControl, FormField, FormItem, FormLabel, FormMessage } from "@/components/ui/form";
import { useToast } from "@/hooks/use-toast";


const phoneRegex = /^\+[1-9]\d{0,2}\s?\d{6,12}$/;

const formSchema = z.object({
  name: z.string().min(2, "Name must be at least 2 characters"),
  email: z.string().email("Invalid email address"),
  phone: z
    .string()
    .regex(
      phoneRegex,
      "Use format: +91 7821882344"
    ),
  subject: z.string().min(5, "Subject must be at least 5 characters"),
  message: z.string().min(10, "Message must be at least 10 characters"),
});

export default function Contact() {
  const { toast } = useToast();
  const form = useForm<z.infer<typeof formSchema>>({
    resolver: zodResolver(formSchema),
    defaultValues: {
      name: "",
      email: "",
      phone: "",
      subject: "",
      message: "",
    },
  });

 async function onSubmit(values: z.infer<typeof formSchema>) {
  try {
    const res = await fetch("/contact", {
      method: "POST",
      headers: {
        "Content-Type": "application/json",
      },
      body: JSON.stringify(values),
    });

    if (!res.ok) {
      throw new Error("Failed to send message");
    }

    toast({
      title: "Message Sent",
      description: "We'll get back to you as soon as possible.",
    });

    form.reset();
  } catch (error) {
    toast({
      title: "Something went wrong",
      description: "Please try again later.",
      variant: "destructive",
    });
  }
}


  const faqs = [
    { q: "Do I need programming experience before enrolling?", a: "No — basic logic understanding helps, but most programs include foundational coding support." },
    { q: "Are sessions live or recorded?", a: "Sessions are live. Recordings are shared after each class for revision and reference." },
    { q: "Will I receive project files or sample code?", a: "Yes — every module includes reference scripts, exercises, and real automation project frameworks." },
    { q: "Do you help with resumes and job preparation?", a: "Yes — resume assistance, project placement, and guidance on interviews are included." },
    { q: "Can companies request customized workflows or projects?", a: "Yes — corporate programs can include organization-specific automation development and code review." },
    { q: "Do I get lifetime access to recordings?", a: "Recordings access remains available for 3-6 months depending on course level." },
    { q: "Is certification provided?", a: "Yes — digital certification is provided after successful completion of the modules and project." },
  ];

  return (
    <div className="min-h-screen flex flex-col bg-background">
      <Navbar />

      <section className="py-20 bg-slate-900 text-white">
        <div className="container mx-auto px-4 text-center">
          <h1 className="font-display text-4xl font-bold mb-4">Get in Touch</h1>
          <p className="text-xl text-slate-300 max-w-2xl mx-auto">
            Have questions about our services or training? Ready to start a project?
          </p>
        </div>
      </section>

      <section className="py-20 bg-white">
        <div className="container mx-auto px-4">
          <div className="grid lg:grid-cols-2 gap-16">
            
            {/* Contact Form */}
            <div>
              <h2 className="font-display text-2xl font-bold text-slate-900 mb-6">Send us a Message</h2>
              <Form {...form}>
                <form onSubmit={form.handleSubmit(onSubmit)} className="space-y-6">
                  <FormField
                    control={form.control}
                    name="name"
                    render={({ field }) => (
                      <FormItem>
                        <FormLabel>Name</FormLabel>
                        <FormControl>
                          <Input placeholder="John Doe" {...field} />
                        </FormControl>
                        <FormMessage />
                      </FormItem>
                    )}
                  />
                  <FormField
                    control={form.control}
                    name="email"
                    render={({ field }) => (
                      <FormItem>
                        <FormLabel>Email</FormLabel>
                        <FormControl>
                          <Input placeholder="john@example.com" {...field} />
                        </FormControl>
                        <FormMessage />
                      </FormItem>
                    )}
                  />


<FormField
  control={form.control}
  name="phone"
  render={({ field }) => (
    <FormItem>
      <FormLabel>Phone Number</FormLabel>
      <FormControl>
        <Input
        placeholder="6699779955"
          {...field}
        />
      </FormControl>
      <FormMessage />
    </FormItem>
  )}
/>



                  <FormField
                    control={form.control}
                    name="subject"
                    render={({ field }) => (
                      <FormItem>
                        <FormLabel>Subject</FormLabel>
                        <FormControl>
                          <Input placeholder="Training Inquiry / Project Request" {...field} />
                        </FormControl>
                        <FormMessage />
                      </FormItem>
                    )}
                  />
                  <FormField
                    control={form.control}
                    name="message"
                    render={({ field }) => (
                      <FormItem>
                        <FormLabel>Message</FormLabel>
                        <FormControl>
                          <Textarea placeholder="Tell us about your requirements..." className="min-h-30" {...field} />
                        </FormControl>
                        <FormMessage />
                      </FormItem>
                    )}
                  />
                  <Button type="submit" className="w-full bg-primary hover:bg-primary/90">
                    Send Message <Send className="ml-2 h-4 w-4" />
                  </Button>
                </form>
              </Form>
            </div>

            {/* Info & FAQ */}
            <div className="space-y-12">
              <div>
                <h2 className="font-display text-2xl font-bold text-slate-900 mb-6">Contact Information</h2>
                <div className="space-y-6">
                  <Card className="border-l-4 border-l-primary shadow-sm">
                    <CardContent className="pt-6 flex items-start gap-4">
                      <Mail className="h-6 w-6 text-primary shrink-0" />
                      <div>
                        <h3 className="font-bold text-slate-900">Email</h3>
                        <a href="mailto:enquiry@cadivity.com" className="text-slate-600 hover:text-primary">enquiry@cadivity.com</a>
                      </div>
                    </CardContent>
                  </Card>
                  <Card className="border-l-4 border-l-accent shadow-sm">
                    <CardContent className="pt-6 flex items-start gap-4">
                      <Phone className="h-6 w-6 text-accent shrink-0" />
                      <div>
                        <h3 className="font-bold text-slate-900">Phone</h3>
                        <a href="tel:+916372495858" className="text-slate-600 hover:text-primary">+91-6372495858</a>
                      </div>
                    </CardContent>
                  </Card>
                </div>
              </div>

              <div>
                <h2 className="font-display text-2xl font-bold text-slate-900 mb-6">Frequently Asked Questions</h2>
                <Accordion type="single" collapsible className="w-full">
                  {faqs.map((faq, i) => (
                    <AccordionItem key={i} value={`item-${i}`}>
                      <AccordionTrigger className="text-left font-medium text-slate-900 hover:text-primary">{faq.q}</AccordionTrigger>
                      <AccordionContent className="text-slate-600">
                        {faq.a}
                      </AccordionContent>
                    </AccordionItem>
                  ))}
                </Accordion>
              </div>
            </div>

          </div>
        </div>
      </section>

      <Footer />
    </div>
  );
}
